/***************************************************************** 
*  @file CRYS_HOST_TST_HASH.c
*  @brief Functions used for HOST-SEP communication to send message from HOST to SEP
*         that contains requested API to be executed in SEP and API parameters
*         Used for SEP External application ATP testing.
*
*  @author mottig
*  @remarks Copyright (C) 2004 by Discretix Technologies Ltd.
*           All Rights reserved
*/



/********************** Include Files *************************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"
#include "DX_VOS_Memmap.h"
#include "DX_VOS_Sem.h"
#include "SEPDriver.h" 
#include "ExtApp_qa_op_code.h"
#include "gen.h"
#include "error.h"
#include "CRYS_HOST_TST_HASH.h"
#include "ExtApp_CRYS_Defines.h"


/************************ Defines ******************************/
/************************ MACROS *******************************/
/************************ Global Data **************************/
/**************** Private function prototype *******************/




/************************ Public Functions ******************************/

 
/***********************************************************************
* Function Name:                                                           
*  SEPQA_HASH_NotIntegrated         
* 
* Inputs:
* Inputs:
* OperationMode - The operation mode for hash
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes HASH not integrated test
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_HASH_NotIntegrated(CRYS_HASH_OperationMode_t   OperationMode,
        								   DxUint8_t                  *DataIn_ptr,
        								   DxUint32_t                  DataInSize,
        								   DxUint8_t                  *ExpectedData_ptr,        
        								   DxUint32_t                  DataExpSize,
        								   char                       *FailureFunctionNameOut_ptr,
        								   DxUint32_t                  FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 3;

   /* parameters */
   DxUint32_t   Host_messageParam[3] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_HASH_NotIntegrated_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_HASH_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_HASH_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_HASH_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
    /* send data in */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send expected data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_HASH_NotIntegrated */

/***********************************************************************
* Function Name:                                                           
*  SEPQA_HASH_Integrated         
* 
* Inputs:
* Inputs:
* OperationMode - The operation mode for hash
* DataIn_ptr - a pointer to the buffer that stores the data to be 
*                       hashed . 
* DataInSize - The size of the data to be hashed in bytes. 
* ExpectedData_ptr - The pointer to the expected data of the AES operation. 
*                   The pointer's value does not need to be word-aligned.
* DataExpSize - The size of the expected data
* OutputLocation - indication if input buffer == output buffer or not.
* FailureFunctionNameOut_ptr - In case that function failures this parameter
*                                             returns the name of the function that failures
* FuncNameMaxSize - The allocated size of the FailureFunctionNameOut_ptr
*
* Outputs:
*  CRYSError_t - Function error return                                     
*
* Description:
*  This function is used to operate the qa external application on the SEP, 
*  that executes HASH integrated test
* Algorithm:       
* 1. initiate and start the message 
* 2. prepare message with all non buffer parameters of the API
* 3. send all buffers params                                   
* 4. end message
* 5. wait for the response of SEP side
* 6. start reading message from the SEP  and check returned function status from SEP
*
**************************************************************************/                                
EXPORT_C DxUint32 SEPQA_HASH_Integrated(CRYS_HASH_OperationMode_t   OperationMode,
        								   DxUint8_t                  *DataIn_ptr,
        								   DxUint32_t                  DataInSize,
        								   DxUint8_t                  *ExpectedData_ptr,        
        								   DxUint32_t                  DataExpSize,
        								   char                       *FailureFunctionNameOut_ptr,
        								   DxUint32_t                  FuncNameMaxSize)
{
   /********* Variables Definitions ************/
   
   /* offset into SRAM */
   DxUint32_t   Host_sramOffset = 0;
   
   /* The return error identifier */
   DxUint32_t 	Host_error = DX_OK;  
   
   /* number of non buffers (simple) parameters */
   DxUint32_t   Host_numOfSimpleParams = 3;

   /* parameters */
   DxUint32_t   Host_messageParam[3] = {0};
   
   /* op code */
   DxUint32_t   Host_opCode = 0;
   	 
  
   /* max length */
   DxUint32_t   Host_maxLength = 0;   
   

   /* set the op code*/
   Host_opCode = DX_SEP_HOST_SEP_PROTOCOL_HOST_HASH_Integrated_TEST_OP_CODE;

   /* 1. initiate and start the message */
   /*************************************/ 
   Host_error = SEPDriver_BuildMsgHeader(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
      
   
   /* 2. prepare message with all non buffer parameters of the API */
   /****************************************************************/
   Host_messageParam[DX_SEP_HOST_HASH_MODE_OFFSET]           = OperationMode;
   Host_messageParam[DX_SEP_HOST_HASH_DATAINSIZE_OFFSET]     = DataInSize;
   Host_messageParam[DX_SEP_HOST_HASH_DATAEXPSIZE_OFFSET]    = DataExpSize;
   
   
   /* write message to shared memory */
   Host_error = SEPDriver_WriteParamater((DxUint32_t)&Host_messageParam[0],
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    sizeof(DxUint32_t)*Host_numOfSimpleParams,
                                    &Host_sramOffset,
                                    DX_FALSE);
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
   /* 3. send all buffers params                                     */
   /* Note that third parameter in function SEPDriver_WriteParamater */
   /* is length of the buffer word aligned                           */
   /******************************************************************/
   
    /* send data in */
   Host_maxLength = ((DataInSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)DataIn_ptr , 
                                     DataInSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }
   
  
                           
   /* send expected data */
   Host_maxLength = ((DataExpSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
   
   Host_error = SEPDriver_WriteParamater((DxUint32_t)ExpectedData_ptr , 
                                     DataExpSize , 
                                     Host_maxLength,
                                     &Host_sramOffset , 
                                     DX_TRUE);
                                     
   if(Host_error != DX_OK)
   {
       goto end_function_unlock;
   }

   /* end message */
   SEPDriver_EndMessage(Host_sramOffset);
            
   /* wait for the response */
   Host_error = SEPDriver_POLL_FOR_REPONSE();
   
   if(Host_error != DX_OK)
   {
     goto end_function_unlock;
   }
   
   /*-------------------------------------
      start reading message from the SEP 
   --------------------------------------*/
   
   /* check message status from SEP */
   Host_error = SEPDriver_CheckStatus(&Host_sramOffset, Host_opCode);
   
   if(Host_error != DX_OK)
   {
       Host_maxLength = ((FuncNameMaxSize + 3) / sizeof(DxUint32_t)) * sizeof(DxUint32_t);
       
       /* read the name of the function that failured */
       SEPDriver_ReadParamater((DxUint32_t)FailureFunctionNameOut_ptr,
                                FuncNameMaxSize,
                                Host_maxLength,
                                &Host_sramOffset,
                                DX_FALSE);
       goto end_function_unlock;
   }
   
   
end_function_unlock:   

   /* lock access to the SEP */
   SEPDriver_Unlock();

   return Host_error;   
   
}/* END OF SEPQA_DES_NotIntegrated */


